// controllers/authController.js
const { poolPromise } = require("../config/db");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");

// 🔐 LOGIN
exports.login = async (req, res) => {
  try {
    const { usuario, contrasena } = req.body;

    if (!usuario || !contrasena) {
      return res.status(400).json({ ok: false, message: "Campos incompletos" });
    }

    const pool = await poolPromise;
    const result = await pool.request()
      .input("usuario", usuario)
      .query(`
        SELECT 
          u.id_usuario, 
          u.nombre_completo, 
          u.usuario, 
          u.contrasena, 
          r.nombre_rol
        FROM Usuarios u
        INNER JOIN Roles r ON u.id_rol = r.id_rol
        WHERE u.usuario = @usuario
      `);

    if (result.recordset.length === 0) {
      return res.status(404).json({ ok: false, message: "Usuario no encontrado" });
    }

    const user = result.recordset[0];

    // Verifica contraseña
    const valid = await bcrypt.compare(contrasena, user.contrasena);
    if (!valid) {
      return res.status(401).json({ ok: false, message: "Contraseña incorrecta" });
    }

    // Crear token JWT
    const token = jwt.sign(
      {
        id_usuario: user.id_usuario,
        nombre_completo: user.nombre_completo,
        rol: user.nombre_rol,
      },
      process.env.JWT_SECRET || "clave_secreta_segura",
      { expiresIn: "8h" }
    );

    res.json({
      ok: true,
      message: "Inicio de sesión exitoso ✅",
      usuario: {
        id_usuario: user.id_usuario,
        nombre_completo: user.nombre_completo,
        rol: user.nombre_rol,
      },
      token,
    });

  } catch (err) {
    console.error("❌ Error en login:", err);
    res.status(500).json({ ok: false, message: "Error al iniciar sesión" });
  }
};


// 🔄 RESTABLECER CONTRASEÑA — CON LOGS DE DEPURACIÓN
exports.resetPassword = async (req, res) => {
  try {
    // 🟦 LOG para ver qué recibe el backend
    console.log("BODY RECIBIDO ===>", req.body);

    const usuario = req.body.usuario;
    const nuevaContrasena =
      req.body.nuevaContrasena || req.body.nueva_contrasena;

    // 🟦 LOG de los datos detectados
    console.log("DATOS QUE LLEGARON:", { usuario, nuevaContrasena });

    if (!usuario || !nuevaContrasena) {
      return res
        .status(400)
        .json({ ok: false, message: "Campos incompletos" });
    }

    // Encriptar contraseña
    const hashed = await bcrypt.hash(nuevaContrasena, 10);

    const pool = await poolPromise;

    // Actualizar contraseña
    await pool.request()
      .input("usuario", usuario)
      .input("contrasena", hashed)
      .query(`
        UPDATE Usuarios
        SET contrasena = @contrasena 
        WHERE usuario = @usuario
      `);

    res.json({
      ok: true,
      message: "Contraseña actualizada correctamente ✅",
    });

  } catch (err) {
    console.error("❌ Error al restablecer contraseña:", err);
    res.status(500).json({
      ok: false,
      message: "Error al restablecer contraseña",
    });
  }
};
