// controllers/incidentesController.js
const { poolPromise } = require("../config/db");


// 📋 Obtener todos los incidentes
exports.getIncidentes = async (req, res) => {
  try {
    const pool = await poolPromise;

    const result = await pool.request().query(`
      SELECT 
        i.id_incidente,
        i.id_activo,
        a.nombre_activo,
        i.descripcion,
        i.impacto,
        i.acciones,
        i.estado,
        CONVERT(VARCHAR, i.fecha_registro, 103) AS fecha_registro
      FROM Incidentes i
      LEFT JOIN Activos a ON i.id_activo = a.id_activo   -- ✔ CAMBIO IMPORTANTE
      ORDER BY i.fecha_registro DESC;
    `);

    res.json(result.recordset);
  } catch (error) {
    console.error("❌ Error al obtener incidentes:", error);
    res.status(500).send("Error al obtener incidentes");
  }
};



// ➕ Crear incidente
exports.createIncidente = async (req, res) => {
  try {
    const { id_activo, descripcion, impacto, acciones, estado, fecha_registro } = req.body;

    // ✔ Ahora NO obligamos id_activo (puede ser null)
    if (!descripcion || !fecha_registro) {
      return res.status(400).send("Faltan datos obligatorios");
    }

    const pool = await poolPromise;

    await pool.request()
      .input("id_activo", id_activo || null)    // ✔ Acepta NULL
      .input("descripcion", descripcion)
      .input("impacto", impacto)
      .input("acciones", acciones)
      .input("estado", estado)
      .input("fecha_registro", new Date(fecha_registro))  // ✔ Fecha correcta
      .query(`
        INSERT INTO Incidentes (id_activo, descripcion, impacto, acciones, estado, fecha_registro)
        VALUES (@id_activo, @descripcion, @impacto, @acciones, @estado, @fecha_registro);
      `);

    res.status(201).send("✅ Incidente creado correctamente");
  } catch (error) {
    console.error("❌ Error al crear incidente:", error);
    res.status(500).send("Error al crear incidente");
  }
};



// ✏️ Actualizar incidente
exports.updateIncidente = async (req, res) => {
  try {
    const { id } = req.params;
    const { id_activo, descripcion, impacto, acciones, estado, fecha_registro } = req.body;

    const pool = await poolPromise;

    await pool.request()
      .input("id_incidente", id)
      .input("id_activo", id_activo || null)     // ✔ Acepta NULL
      .input("descripcion", descripcion)
      .input("impacto", impacto)
      .input("acciones", acciones)
      .input("estado", estado)
      .input("fecha_registro", new Date(fecha_registro))   // ✔ Mantiene la fecha
      .query(`
        UPDATE Incidentes
        SET 
          id_activo = @id_activo,
          descripcion = @descripcion,
          impacto = @impacto,
          acciones = @acciones,
          estado = @estado,
          fecha_registro = @fecha_registro
        WHERE id_incidente = @id_incidente;
      `);

    res.send("✏️ Incidente actualizado correctamente");
  } catch (error) {
    console.error("❌ Error al actualizar incidente:", error);
    res.status(500).send("Error al actualizar incidente");
  }
};



// 🗑️ Eliminar incidente
exports.deleteIncidente = async (req, res) => {
  try {
    const { id } = req.params;

    const pool = await poolPromise;

    await pool.request()
      .input("id_incidente", id)
      .query(`DELETE FROM Incidentes WHERE id_incidente = @id_incidente`);

    res.send("🗑️ Incidente eliminado correctamente");
  } catch (error) {
    console.error("❌ Error al eliminar incidente:", error);
    res.status(500).send("Error al eliminar incidente");
  }
};
