// middleware/authMiddleware.js
// PEGA ESTE CÓDIGO EN EL ARCHIVO NUEVO

const jwt = require("jsonwebtoken");

// 🔥 IMPORTANTE: Asegúrate de tener esta variable en tu archivo .env
// Es la "llave secreta" con la que firmas el token al hacer login
const JWT_SECRET = process.env.JWT_SECRET; 

if (!JWT_SECRET) {
  console.error("Error fatal: La variable de entorno JWT_SECRET no está definida.");
  process.exit(1); // Detiene la aplicación si la variable no existe
}

exports.verificarToken = (req, res, next) => {
  try {
    const authHeader = req.headers["authorization"];
    
    // Verificar que el header 'Authorization' exista y tenga el formato 'Bearer <token>'
    if (!authHeader || !authHeader.startsWith("Bearer ")) {
      return res.status(401).json({ error: "Acceso denegado. Formato de token inválido o no proporcionado." });
    }

    const token = authHeader.split(" ")[1]; // Extraer solo el token

    if (!token) {
      return res.status(401).json({ error: "Acceso denegado. No se proporcionó token." });
    }

    // Verificar el token con la llave secreta
    const decodificado = jwt.verify(token, JWT_SECRET);

    // Asumimos que al crear el token (en tu login) guardaste el rol así:
    // jwt.sign({ id: usuario.id, rol: usuario.nombre_rol }, JWT_SECRET)
    // 'decodificado' ahora es un objeto como: { id: 1, rol: 'Administrador' }
    
    if (!decodificado.rol) {
         return res.status(403).json({ error: "Token inválido. El rol no está incluido en el token." });
    }
    
    req.usuario = decodificado; // Adjuntamos los datos del usuario al objeto 'req'
    
    next(); // El token es válido, pasa al siguiente middleware (permisosMiddleware)

  } catch (error) {
    // El token es inválido o expiró
    if (error.name === 'JsonWebTokenError') {
      return res.status(400).json({ error: "Token inválido." });
    }
    if (error.name === 'TokenExpiredError') {
      return res.status(401).json({ error: "Token expirado." });
    }
    // Otro error
    console.error("Error en authMiddleware:", error);
    return res.status(500).json({ error: "Error interno al verificar el token." });
  }
};